const DEFAULTS = {
  contentSelector: ".prose",
  headlineSelector: "h4"
};

const SESSION_DEFAULTS = {
  active: false,
  tabId: null,
  lastUrl: null,
  items: [] // [{ url, title, contentHtml, contentText, paragraphsText, capturedAt }]
};

// --- Storage helpers ---
async function getStoredSettings() {
  const stored = await chrome.storage.sync.get(DEFAULTS);
  return {
    contentSelector: (stored.contentSelector || "").trim() || DEFAULTS.contentSelector,
    headlineSelector: (stored.headlineSelector || "").trim() || DEFAULTS.headlineSelector
  };
}

async function getSession() {
  const stored = await chrome.storage.local.get(SESSION_DEFAULTS);
  return {
    active: !!stored.active,
    tabId: stored.tabId ?? null,
    lastUrl: stored.lastUrl ?? null,
    items: Array.isArray(stored.items) ? stored.items : []
  };
}

async function setSession(patch) {
  const current = await getSession();
  const next = { ...current, ...patch };
  await chrome.storage.local.set(next);
  return next;
}

function isoNow() {
  return new Date().toISOString();
}

async function showBadgeCount(count) {
  try {
    await chrome.action.setBadgeText({ text: count ? String(count) : "" });
  } catch {
    // ignore
  }
}

// Toast in the page (nice UX)
async function injectToast(tabId, message) {
  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: (msg) => {
        const id = "__bulk_copy_toast__";
        let el = document.getElementById(id);

        if (!el) {
          el = document.createElement("div");
          el.id = id;
          el.style.position = "fixed";
          el.style.zIndex = "2147483647";
          el.style.right = "12px";
          el.style.bottom = "12px";
          el.style.padding = "10px 12px";
          el.style.borderRadius = "10px";
          el.style.boxShadow = "0 6px 20px rgba(0,0,0,.25)";
          el.style.background = "rgba(20,20,20,.92)";
          el.style.color = "#fff";
          el.style.fontSize = "12px";
          el.style.fontFamily =
            "system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif";
          el.style.maxWidth = "60vw";
          el.style.lineHeight = "1.3";
          document.documentElement.appendChild(el);
        }

        el.textContent = msg;
        el.style.opacity = "1";
        clearTimeout(window.__bulk_copy_toast_timer__);
        window.__bulk_copy_toast_timer__ = setTimeout(() => {
          el.style.opacity = "0";
        }, 1800);
      },
      args: [message]
    });
  } catch {
    // Some pages may block scripting; ignore quietly
  }
}

// Capture function executed inside the page
function captureFromPage(selectors) {
  const { contentSelector, headlineSelector } = selectors;

  const result = {
    ok: false,
    message: "",
    url: location.href,
    title: "",
    contentHtml: "",
    contentText: "",
    paragraphsText: []
  };

  try {
    // ✅ IMPORTANT: define helper INSIDE injected function
    function getFirstSpanTextFromParagraph(pEl) {
      const firstSpan = pEl ? pEl.querySelector("span") : null;
      if (!firstSpan) return "";
      return (firstSpan.textContent || "").trim();
    }

    let headlineEl = null;
    let contentEl = null;

    try {
      headlineEl = document.querySelector(headlineSelector);
    } catch (e) {
      result.message = `Invalid headline selector: ${headlineSelector}\n${String(e)}`;
      return result;
    }

    try {
      contentEl = document.querySelector(contentSelector);
    } catch (e) {
      result.message = `Invalid content selector: ${contentSelector}\n${String(e)}`;
      return result;
    }

    if (!headlineEl) {
      result.message = `No headline found for: "${headlineSelector}"`;
      return result;
    }
    if (!contentEl) {
      result.message = `No content found for: "${contentSelector}"`;
      return result;
    }

    const title = (headlineEl.innerText || headlineEl.textContent || "").trim();
    const contentText = (contentEl.innerText || contentEl.textContent || "").trim();
    const contentHtml = (contentEl.innerHTML || "").trim();

    if (!title) {
      result.message = `Headline "${headlineSelector}" has no text.`;
      return result;
    }

    // Direct child <p> tags only
    const paragraphsText = Array.from(contentEl.children)
      .filter((el) => el && el.tagName && el.tagName.toLowerCase() === "p")
      .map((p) => getFirstSpanTextFromParagraph(p))
      .map((t) => String(t || "").trim())
      .filter(Boolean);

    if (!paragraphsText.length && !contentText && !contentHtml) {
      result.message = `Content "${contentSelector}" is empty.`;
      return result;
    }

    result.ok = true;
    result.title = title;
    result.contentText = contentText;
    result.contentHtml = contentHtml;
    result.paragraphsText = paragraphsText;
    result.message = paragraphsText.length
      ? `Captured ${paragraphsText.length} paragraphs (first-span text only).`
      : "Captured.";

    return result;
  } catch (e) {
    // ✅ ensures we never return "Unknown error"
    result.ok = false;
    result.message = `captureFromPage exception: ${String(e)}`;
    return result;
  }
}

// Main capture pipeline
async function attemptCapture(tabId, urlMaybe) {
  const session = await getSession();
  if (!session.active || session.tabId !== tabId) return;

  // Get freshest URL
  let url = urlMaybe;
  try {
    const tab = await chrome.tabs.get(tabId);
    url = tab.url || url;
  } catch {}

  if (!url) return;

  // Prevent duplicate captures for same URL
  if (session.lastUrl && session.lastUrl === url) return;

  const selectors = await getStoredSettings();

  let pageResult;
  try {
    const injected = await chrome.scripting.executeScript({
      target: { tabId },
      func: captureFromPage,
      args: [selectors]
    });

    pageResult = injected?.[0]?.result;
  } catch (e) {
    await injectToast(tabId, `Capture failed (executeScript). ${String(e)}`);
    return;
  }

  if (!pageResult?.ok) {
    const msg = pageResult?.message ? String(pageResult.message) : "No result returned from page.";
    await injectToast(tabId, `Not captured: ${msg}`);
    await setSession({ lastUrl: url });
    return;
  }

  const nextItems = [
    ...(session.items || []),
    {
      url: pageResult.url || url,
      title: pageResult.title,
      contentHtml: pageResult.contentHtml,
      contentText: pageResult.contentText,
      paragraphsText: Array.isArray(pageResult.paragraphsText) ? pageResult.paragraphsText : [],
      capturedAt: isoNow()
    }
  ];

  const next = await setSession({
    items: nextItems,
    lastUrl: url
  });

  await showBadgeCount(next.items.length);

  const n = pageResult.paragraphsText?.length || 0;
  await injectToast(
    tabId,
    `Captured #${next.items.length}: ${pageResult.title}${n ? ` (${n} paragraphs)` : ""}`
  );
}

// --- Listen for tab updates (navigation complete) ---
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.status !== "complete") return;
  await attemptCapture(tabId, tab.url);
});

// --- Handle Begin / End / Get requests from popup ---
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    if (msg?.type === "BEGIN_SESSION") {
      const tabId = msg.tabId;
      if (!tabId) {
        sendResponse({ ok: false, message: "No active tabId provided." });
        return;
      }

      await setSession({
        active: true,
        tabId,
        lastUrl: null,
        items: []
      });

      await showBadgeCount(0);

      // Immediately attempt capture on current page
      await attemptCapture(tabId);

      sendResponse({ ok: true });
      return;
    }

    if (msg?.type === "END_SESSION") {
      const session = await setSession({
        active: false,
        tabId: null,
        lastUrl: null
      });

      await showBadgeCount(0);

      sendResponse({ ok: true, count: session.items.length });
      return;
    }

    if (msg?.type === "GET_SESSION") {
      const session = await getSession();
      sendResponse({ ok: true, session });
      return;
    }

    sendResponse({ ok: false, message: "Unknown message type." });
  })();

  return true;
});
