/**
 * popup.js (Bulk Copy Prose / Headline)
 *
 * UPDATE:
 * - Chapter parsing now supports BOTH formats:
 *    1) "Chapter 695: Title"
 *    2) "Chapter 695 (1): Title"  (multi-part chapters)
 * - Validation still enforces NON-DECREASING chapter numbers in capture order.
 *   (695 (1) then 695 (2) is allowed because chapter number = 695 for both.)
 */

const DEFAULTS = {
  contentSelector: ".prose",
  headlineSelector: "h4"
};

const EXPORT_DEFAULTS = {
  exportStartTimestamp: "" // stored as ISO string
};

const statusEl = document.getElementById("status");
const sessionInfoEl = document.getElementById("sessionInfo");
const downloadAreaEl = document.getElementById("downloadArea");

const btnBegin = document.getElementById("beginSession");
const btnEnd = document.getElementById("endSession");
const btnExport = document.getElementById("exportWxr");

const inputContentSelector = document.getElementById("contentSelector");
const inputHeadlineSelector = document.getElementById("headlineSelector");
const btnSaveSettings = document.getElementById("saveSettings");
const btnResetSettings = document.getElementById("resetSettings");
const settingsStatusEl = document.getElementById("settingsStatus");

// Export timestamp UI (from prior update)
const inputStartTimestamp = document.getElementById("startTimestamp");
const btnUseNow = document.getElementById("useNow");

/* ---------------- Tabs ---------------- */
document.querySelectorAll(".tab").forEach((tab) => {
  tab.addEventListener("click", () => {
    document.querySelectorAll(".tab").forEach((t) => t.classList.remove("active"));
    document.querySelectorAll(".panel").forEach((p) => p.classList.remove("active"));

    tab.classList.add("active");
    const which = tab.dataset.tab;
    document.getElementById(`panel-${which}`).classList.add("active");
  });
});

/* -------------- Helpers -------------- */
function setStatus(msg, type = "ok") {
  statusEl.className = type;
  statusEl.textContent = msg;
}

function setSettingsStatus(msg) {
  settingsStatusEl.textContent = msg;
}

function escapeXml(str) {
  return String(str)
    .replaceAll("&", "&amp;")
    .replaceAll("<", "&lt;")
    .replaceAll(">", "&gt;")
    .replaceAll('"', "&quot;")
    .replaceAll("'", "&apos;");
}

function escapeHtml(str) {
  return String(str)
    .replaceAll("&", "&amp;")
    .replaceAll("<", "&lt;")
    .replaceAll(">", "&gt;")
    .replaceAll('"', "&quot;")
    .replaceAll("'", "&#039;");
}

function slugify(str) {
  return String(str)
    .trim()
    .toLowerCase()
    .replace(/&/g, "and")
    .replace(/[^a-z0-9]+/g, "-")
    .replace(/^-+|-+$/g, "");
}

/* ---- datetime-local helpers ---- */
function toDatetimeLocalValue(date) {
  const pad = (n) => String(n).padStart(2, "0");
  return (
    date.getFullYear() +
    "-" +
    pad(date.getMonth() + 1) +
    "-" +
    pad(date.getDate()) +
    "T" +
    pad(date.getHours()) +
    ":" +
    pad(date.getMinutes())
  );
}

function parseDatetimeLocalValue(val) {
  if (!val) return null;
  const d = new Date(val);
  return Number.isFinite(d.getTime()) ? d : null;
}

function wpDateFromDate(d) {
  const pad = (n) => String(n).padStart(2, "0");
  return (
    d.getFullYear() +
    "-" +
    pad(d.getMonth() + 1) +
    "-" +
    pad(d.getDate()) +
    " " +
    pad(d.getHours()) +
    ":" +
    pad(d.getMinutes()) +
    ":" +
    pad(d.getSeconds())
  );
}

/* -------------- Chapter parsing + validation (UPDATED) -------------- */
function parseChapterNumber(title) {
  // Supports:
  // - "Chapter 695: ..."
  // - "Chapter 695 (1): ..."
  // - "chapter 695(2): ..." (optional spacing)
  //
  // Captures the main chapter number (695). Part numbers are ignored for ordering validation.
  const m = String(title || "").match(/chapter\s+(\d+)\s*(?:\(\s*\d+\s*\))?\s*:/i);
  if (!m) return null;
  const n = Number(m[1]);
  return Number.isFinite(n) ? n : null;
}

function validateCaptureOrderChaptersNonDecreasing(items) {
  const problems = [];
  let prev = null;

  items.forEach((it, idx) => {
    const title = it?.title || "";
    const ch = parseChapterNumber(title);

    if (!Number.isFinite(ch)) {
      problems.push(
        `#${idx + 1}: Missing/invalid chapter number in title (expected "Chapter <num>: ..." OR "Chapter <num> (<part>): ..."): "${title}"`
      );
      return;
    }

    if (prev !== null && ch < prev) {
      problems.push(
        `#${idx + 1}: Chapter ${ch} is less than previous chapter ${prev} (title: "${title}")`
      );
    }

    prev = ch;
  });

  return problems;
}

/* -------------- Settings Storage -------------- */
async function getSettings() {
  const stored = await chrome.storage.sync.get(DEFAULTS);
  return {
    contentSelector: (stored.contentSelector || "").trim() || DEFAULTS.contentSelector,
    headlineSelector: (stored.headlineSelector || "").trim() || DEFAULTS.headlineSelector
  };
}

async function saveSettings(contentSelector, headlineSelector) {
  await chrome.storage.sync.set({
    contentSelector: (contentSelector || "").trim(),
    headlineSelector: (headlineSelector || "").trim()
  });
}

async function loadSettingsIntoInputs() {
  const raw = await chrome.storage.sync.get(DEFAULTS);

  inputContentSelector.value = (raw.contentSelector || "").trim();
  inputHeadlineSelector.value = (raw.headlineSelector || "").trim();

  const effective = await getSettings();
  setSettingsStatus(
    `Effective selectors:\nContent: ${effective.contentSelector}\nHeadline: ${effective.headlineSelector}`
  );
}

/* -------------- Export timestamp storage -------------- */
async function getExportPrefs() {
  const stored = await chrome.storage.local.get(EXPORT_DEFAULTS);
  return {
    exportStartTimestamp: stored.exportStartTimestamp || ""
  };
}

async function setExportPrefs(patch) {
  const current = await getExportPrefs();
  const next = { ...current, ...patch };
  await chrome.storage.local.set(next);
  return next;
}

async function loadExportPrefsIntoUi() {
  const prefs = await getExportPrefs();

  if (prefs.exportStartTimestamp) {
    const d = new Date(prefs.exportStartTimestamp);
    if (Number.isFinite(d.getTime())) {
      inputStartTimestamp.value = toDatetimeLocalValue(d);
    }
  } else {
    const d = new Date();
    d.setSeconds(0, 0);
    inputStartTimestamp.value = toDatetimeLocalValue(d);
  }
}

async function saveExportStartFromUi() {
  const d = parseDatetimeLocalValue(inputStartTimestamp.value);
  if (!d) throw new Error("Invalid start timestamp. Please choose a valid date/time.");
  d.setSeconds(0, 0);
  await setExportPrefs({ exportStartTimestamp: d.toISOString() });
  return d;
}

/* -------------- Session (background) -------------- */
async function getSession() {
  return await chrome.runtime.sendMessage({ type: "GET_SESSION" });
}

async function beginSession(tabId) {
  return await chrome.runtime.sendMessage({ type: "BEGIN_SESSION", tabId });
}

async function endSession() {
  return await chrome.runtime.sendMessage({ type: "END_SESSION" });
}

function clearDownloadLink() {
  downloadAreaEl.innerHTML = "";
}

function createDownloadLink(filename, content, mime = "application/xml") {
  const blob = new Blob([content], { type: mime });
  const url = URL.createObjectURL(blob);

  const a = document.createElement("a");
  a.className = "download";
  a.href = url;
  a.download = filename;
  a.textContent = `Download ${filename}`;

  const wrap = document.createElement("div");
  wrap.appendChild(a);

  a.addEventListener("click", () => {
    setTimeout(() => URL.revokeObjectURL(url), 10_000);
  });

  downloadAreaEl.innerHTML = "";
  downloadAreaEl.appendChild(wrap);
}

/* -------------- Gutenberg blocks -------------- */
function ensureBlockWrappedParagraphText(text) {
  const safe = escapeHtml(String(text || ""));
  return `<!-- wp:paragraph --><p>${safe}</p><!-- /wp:paragraph -->`;
}

function blocksFromCapturedParagraphText(item) {
  const paras = Array.isArray(item.paragraphsText) ? item.paragraphsText : [];
  const cleaned = paras.map((t) => String(t || "").trim()).filter(Boolean);
  if (!cleaned.length) return "";
  return cleaned.map(ensureBlockWrappedParagraphText).join("\n\n");
}

/* -------------- Fallback blocks -------------- */
function splitIntoParagraphsFromText(text) {
  const t = String(text || "").replace(/\r\n/g, "\n").replace(/\r/g, "\n").trim();
  if (!t) return [];
  return t
    .split(/\n{2,}/g)
    .map((p) => p.replace(/\n+/g, " ").trim())
    .filter(Boolean);
}

function htmlToTextRough(html) {
  const h = String(html || "");
  if (!h.trim()) return "";

  const withBreaks = h
    .replace(/<\s*br\s*\/?\s*>/gi, "\n")
    .replace(/<\/\s*(p|div|section|article|li|h1|h2|h3|h4|h5|h6)\s*>/gi, "\n\n")
    .replace(/<\s*li\s*>/gi, "• ");

  const stripped = withBreaks.replace(/<[^>]+>/g, "");

  return stripped
    .replace(/&nbsp;/g, " ")
    .replace(/&amp;/g, "&")
    .replace(/&lt;/g, "<")
    .replace(/&gt;/g, ">")
    .replace(/&quot;/g, '"')
    .replace(/&#039;/g, "'")
    .trim();
}

function makeFallbackParagraphBlocks(item) {
  const baseText =
    (item.contentText || "").trim() ||
    htmlToTextRough(item.contentHtml || "");

  const paras = splitIntoParagraphsFromText(baseText);

  if (!paras.length) {
    return `<!-- wp:paragraph --><p></p><!-- /wp:paragraph -->`;
  }

  return paras
    .map((p) => `<!-- wp:paragraph --><p>${escapeHtml(p)}</p><!-- /wp:paragraph -->`)
    .join("\n\n");
}

function makeGutenbergContent(item) {
  const fromTextParas = blocksFromCapturedParagraphText(item);
  if (fromTextParas) return fromTextParas;
  return makeFallbackParagraphBlocks(item);
}

/* -------------- WXR Builder -------------- */
function buildWxrXml(items, categories, startDate) {
  const orderedItems = [...items];

  const problems = validateCaptureOrderChaptersNonDecreasing(orderedItems);
  if (problems.length) {
    const msg =
      "Export blocked: chapter order/title format problems detected.\n\n" +
      problems.slice(0, 25).join("\n") +
      (problems.length > 25 ? `\n…plus ${problems.length - 25} more` : "");
    throw new Error(msg);
  }

  const cats = categories.map((c) => c.trim()).filter(Boolean);
  const uniqueCats = Array.from(new Set(cats));

  const categoryDefs = uniqueCats
    .map((name) => {
      const nicename = slugify(name) || "category";
      return `
  <wp:category>
    <wp:term_id>0</wp:term_id>
    <wp:category_nicename>${escapeXml(nicename)}</wp:category_nicename>
    <wp:category_parent></wp:category_parent>
    <wp:cat_name><![CDATA[${name}]]></wp:cat_name>
  </wp:category>`;
    })
    .join("");

  const channelPubDate = new Date(
    startDate.getTime() + Math.max(0, orderedItems.length - 1) * 60_000
  );

  const itemsXml = orderedItems
    .map((item, idx) => {
      const title = item.title || `Imported Post ${idx + 1}`;
      const link = item.url || "";

      const d = new Date(startDate.getTime() + idx * 60_000);
      const postDate = wpDateFromDate(d);

      const catsXml = uniqueCats
        .map((name) => {
          const nicename = slugify(name) || "category";
          return `<category domain="category" nicename="${escapeXml(nicename)}"><![CDATA[${name}]]></category>`;
        })
        .join("\n      ");

      const content = makeGutenbergContent(item);

      return `
  <item>
    <title><![CDATA[${title}]]></title>
    <link>${escapeXml(link)}</link>
    <pubDate>${escapeXml(d.toUTCString())}</pubDate>
    <dc:creator><![CDATA[bulk-import]]></dc:creator>
    <guid isPermaLink="false">bulk-import-${idx + 1}</guid>
    <description></description>
    <content:encoded><![CDATA[${content}]]></content:encoded>
    <excerpt:encoded><![CDATA[]]></excerpt:encoded>
    ${catsXml ? "      " + catsXml : ""}
    <wp:post_id>${idx + 1}</wp:post_id>
    <wp:post_date><![CDATA[${postDate}]]></wp:post_date>
    <wp:post_date_gmt><![CDATA[${postDate}]]></wp:post_date_gmt>
    <wp:comment_status><![CDATA[closed]]></wp:comment_status>
    <wp:ping_status><![CDATA[closed]]></wp:ping_status>
    <wp:post_name><![CDATA[${slugify(title) || `imported-post-${idx + 1}`}]]></wp:post_name>
    <wp:status><![CDATA[publish]]></wp:status>
    <wp:post_parent>0</wp:post_parent>
    <wp:menu_order>0</wp:menu_order>
    <wp:post_type><![CDATA[post]]></wp:post_type>
    <wp:post_password><![CDATA[]]></wp:post_password>
    <wp:is_sticky>0</wp:is_sticky>
  </item>`;
    })
    .join("\n");

  return `<?xml version="1.0" encoding="UTF-8" ?>
<rss version="2.0"
  xmlns:excerpt="http://wordpress.org/export/1.2/excerpt/"
  xmlns:content="http://purl.org/rss/1.0/modules/content/"
  xmlns:wfw="http://wellformedweb.org/CommentAPI/"
  xmlns:dc="http://purl.org/dc/elements/1.1/"
  xmlns:wp="http://wordpress.org/export/1.2/"
>
<channel>
  <title><![CDATA[Bulk Import]]></title>
  <link>https://example.com</link>
  <description><![CDATA[Bulk imported posts]]></description>
  <pubDate>${escapeXml(channelPubDate.toUTCString())}</pubDate>
  <language>en</language>
  <wp:wxr_version>1.2</wp:wxr_version>

  ${categoryDefs}

  ${itemsXml}
</channel>
</rss>`;
}

/* -------------- UI State -------------- */
async function refreshSessionUi() {
  const res = await getSession();
  if (!res?.ok) return;

  const session = res.session;
  const count = session.items?.length || 0;

  sessionInfoEl.textContent =
    `Session: ${session.active ? "ACTIVE" : "inactive"}\n` +
    `Captured posts: ${count}\n` +
    (session.active ? `Capturing in tabId: ${session.tabId}` : "");

  btnExport.disabled = session.active || count === 0;
  btnBegin.disabled = session.active;
  btnEnd.disabled = !session.active;
}

/* -------------- Button Actions -------------- */
btnBegin.addEventListener("click", async () => {
  try {
    clearDownloadLink();

    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab?.id) throw new Error("No active tab found.");

    const resp = await beginSession(tab.id);

    if (!resp?.ok) {
      setStatus(resp?.message || "Failed to begin session.", "err");
      return;
    }

    setStatus(
      "Copy session started.\nCaptured current page (if selectors matched).\nNow navigate page-by-page; it will auto-capture after each page load.",
      "ok"
    );
    await refreshSessionUi();
  } catch (e) {
    setStatus(String(e), "err");
  }
});

btnEnd.addEventListener("click", async () => {
  try {
    const resp = await endSession();
    if (!resp?.ok) {
      setStatus(resp?.message || "Failed to end session.", "err");
      return;
    }

    clearDownloadLink();
    setStatus(`Session ended. Captured ${resp.count || 0} posts. You can now export WXR.`, "ok");
    await refreshSessionUi();
  } catch (e) {
    setStatus(String(e), "err");
  }
});

btnUseNow?.addEventListener("click", async () => {
  try {
    const d = new Date();
    d.setSeconds(0, 0);
    inputStartTimestamp.value = toDatetimeLocalValue(d);
    await setExportPrefs({ exportStartTimestamp: d.toISOString() });
    setStatus(`Start timestamp set to now: ${toDatetimeLocalValue(d)}`, "ok");
  } catch (e) {
    setStatus(String(e), "err");
  }
});

inputStartTimestamp?.addEventListener("change", async () => {
  try {
    await saveExportStartFromUi();
    setStatus("Start timestamp saved.", "ok");
  } catch (e) {
    setStatus(String(e), "err");
  }
});

btnExport.addEventListener("click", async () => {
  try {
    clearDownloadLink();

    const res = await getSession();
    const session = res?.session;
    const items = session?.items || [];

    if (!items.length) {
      setStatus("No captured posts to export.", "err");
      return;
    }
    if (session.active) {
      setStatus("End the session before exporting.", "err");
      return;
    }

    const startDate = await saveExportStartFromUi();

    const catInput = window.prompt(
      "Enter categories to apply to ALL posts (comma-separated). Leave blank for none.",
      "Imported"
    );

    const categories = (catInput || "")
      .split(",")
      .map((s) => s.trim())
      .filter(Boolean);

    const xml = buildWxrXml(items, categories, startDate);

    const filename = `bulk-import-${new Date().toISOString().slice(0, 10)}.xml`;
    createDownloadLink(filename, xml, "application/xml");

    const firstCh = parseChapterNumber(items[0]?.title);
    const lastCh = parseChapterNumber(items[items.length - 1]?.title);

    setStatus(
      `WXR generated for ${items.length} posts (PUBLISHED).\n` +
        `Order: capture order${firstCh != null && lastCh != null ? ` (Chapter ${firstCh} to Chapter ${lastCh})` : ""}\n` +
        `Start time: ${toDatetimeLocalValue(startDate)}\n` +
        `End time: ${toDatetimeLocalValue(new Date(startDate.getTime() + (items.length - 1) * 60_000))}\n` +
        `Categories: ${categories.length ? categories.join(", ") : "(none)"}\n` +
        `Click the download link below.`,
      "ok"
    );
  } catch (e) {
    setStatus(String(e), "err");
  }
});

/* -------- Settings buttons -------- */
btnSaveSettings.addEventListener("click", async () => {
  try {
    await saveSettings(inputContentSelector.value, inputHeadlineSelector.value);
    await loadSettingsIntoInputs();
    setSettingsStatus(settingsStatusEl.textContent + "\n\nSaved ✅");
  } catch (e) {
    setSettingsStatus("Save failed: " + String(e));
  }
});

btnResetSettings.addEventListener("click", async () => {
  try {
    await chrome.storage.sync.set({
      contentSelector: "",
      headlineSelector: ""
    });
    await loadSettingsIntoInputs();
    setSettingsStatus(settingsStatusEl.textContent + "\n\nReset to defaults ✅");
  } catch (e) {
    setSettingsStatus("Reset failed: " + String(e));
  }
});

/* -------- Init -------- */
loadSettingsIntoInputs();
refreshSessionUi();
loadExportPrefsIntoUi();
